/*
 * cPbcastNode.cpp
 *
 * A pbcast node.
 */
#include "cPbcastNode.h"
#include "Endpoint/cEndpoint.h"
#include "Endpoint/cEndpointFactory.h"
#include "Util/gError.h"
#include <memory.h>


cPbcastNode::cPbcastNode(cEndpoint *ep)
{
	mSeenFirstMsg	= false;
	mEndpoint		= ep;
	mHeartbeat		= 0;
	mGossipSeqNum	= 0;
	mLastDeliveredSeqNum = 0;
	mEndpoint->AddRef();
}

cPbcastNode::cPbcastNode()
{
	mSeenFirstMsg	= false;
	mEndpoint		= NULL;
	mHeartbeat		= 0;
	mGossipSeqNum	= 0;
	mLastDeliveredSeqNum = 0;
}

cPbcastNode::~cPbcastNode()
{
	if(mEndpoint)
	{
		mEndpoint->Release();
		mEndpoint = NULL;
	}
}

int cPbcastNode::GetMinSerialSize()
{
	return (sizeof(u_short) + sizeof(u_long));
}

unsigned int cPbcastNode::HashCode()
{
	return mEndpoint->HashCode();
}

/*
 * cPbcastNode::Print()
 *
 * Purpose:	Prints out the pbcast node to the given stream (for debug purposes)
 * IN:		errStream	-> The stream to use for printing.
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	-
 */
void cPbcastNode::Print(ostream &errStream)
{
	errStream << "+++++++++++++++++++" << endl;
	mEndpoint->Print(errStream);
	errStream << "Heartbeat:  " << mHeartbeat << endl;
	errStream << "GossipSeq:  " << mGossipSeqNum << endl;
	errStream << "LastDelSeq: " << mLastDeliveredSeqNum << endl;
	errStream << "+++++++++++++++++++" << endl;
}

char* cPbcastNode::MinSerialize(cPbcastNode* node, char* buffer, int* size)
{
	int sizeNeeded = sizeof(node->mHeartbeat) + sizeof(node->mLastDeliveredSeqNum);
	if(*size < sizeNeeded)
	{
		*size = sizeNeeded;
		return NULL;
	}

	// Marshal elements
	u_short tempHeartbeat = htons(node->mHeartbeat);
	u_long tempLastDeliver = htonl(node->mLastDeliveredSeqNum);
	memcpy(buffer, &tempHeartbeat, sizeof(tempHeartbeat));
	buffer += sizeof(tempHeartbeat);
	memcpy(buffer, &tempLastDeliver, sizeof(tempLastDeliver));
	buffer += sizeof(tempLastDeliver);
	*size = *size - sizeNeeded;
	return buffer;
}

char* cPbcastNode::MinDeserialize(cPbcastNode* node, char* buffer, int* size)
{
	int sizeNeeded = sizeof(node->mHeartbeat) + sizeof(node->mLastDeliveredSeqNum);
	if(*size < sizeNeeded)
	{
		return NULL;
	}
	memcpy(&(node->mHeartbeat), buffer, sizeof(node->mHeartbeat));
	node->mHeartbeat = ntohs(node->mHeartbeat);		// Undo marshaling
	buffer += sizeof(node->mHeartbeat);
	memcpy(&(node->mLastDeliveredSeqNum), buffer, sizeof(node->mLastDeliveredSeqNum));
	node->mLastDeliveredSeqNum = ntohl(node->mLastDeliveredSeqNum); // Undo marshaling
	buffer += sizeof(node->mLastDeliveredSeqNum);
	*size = *size - sizeNeeded;
	return buffer;
}


/*
 * cPbcastNode::Serialize()
 *
 * Purpose:	Turns the pbcastnode into a byte stream from in the given buffer.
 * IN:		buffer		-> The buffer in which to place serialized endpoint.
 *			size		-> The size of the buffer provided.
 * OUT:		size		-> The size remaining in the buffer, or the size needed if fail
 * Cond:	-
 * PostCnd:	The endpoint is in the buffer.
 * Return:	The address of the location in the buffer after the serialized representation
 *			OR NULL if failure.
 */
char* cPbcastNode::Serialize(char* buffer, int *size)
{
	int	tempSize = *size;
	int sizeNeeded = GetSize();

	// Marshal elements
	u_short tempHeartbeat = htons(mHeartbeat);
	u_long tempLastDeliver = htonl(mLastDeliveredSeqNum);

	if(*size < sizeNeeded)
	{
		*size = sizeNeeded;
		return NULL;
	}
	buffer = mEndpoint->Serialize(buffer, &tempSize);
	memcpy(buffer, &tempHeartbeat, sizeof(tempHeartbeat));
	buffer += sizeof(tempHeartbeat);
	memcpy(buffer, &tempLastDeliver, sizeof(tempLastDeliver));
	buffer += sizeof(tempLastDeliver);
	*size = *size - sizeNeeded;
	return buffer;
}

/*
 * cPbcastNode::Deserialize()
 *
 * Purpose:	Fills in the node information from the given buffer.
 * IN:		buffer		-> The buffer in which the endpoint info resides.
 *			size		-> The size of the buffer provided.
 * OUT:		size		-> The size remaining in the buffer, or the size needed if fail
 * Cond:	-
 * PostCnd:	This nodeinfo is set up
 * Return:	The address of the location in the buffer after the serialized representation
 *			OR NULL if failure.
 */
char* cPbcastNode::Deserialize(char *buffer, int *size)
{
	if(mEndpoint)
	{
		mEndpoint->Release();	// Free up endpoint.
	}
	mEndpoint = cEndpointFactory::AllocEndpoint(&buffer, size);
	if(!mEndpoint)
	{
		gError("Unable to deserialize an endpoint into a nodeinfo.", __LINE__, __FILE__);
	}
	mEndpoint->AddRef();
	memcpy(&mHeartbeat, buffer, sizeof(mHeartbeat));
	mHeartbeat = ntohs(mHeartbeat);		// Undo marshaling
	buffer += sizeof(mHeartbeat);
	memcpy(&mLastDeliveredSeqNum, buffer, sizeof(mLastDeliveredSeqNum));
	mLastDeliveredSeqNum = ntohl(mLastDeliveredSeqNum); // Undo marshaling
	buffer += sizeof(mLastDeliveredSeqNum);
	*size = *size - sizeof(mHeartbeat);		// CHANGE!
	return buffer;
}

unsigned int cPbcastNode::GetSize()
{
	return sizeof(mHeartbeat) + sizeof(mLastDeliveredSeqNum) + mEndpoint->GetSize();
}

bool cPbcastNode::Equals(cHashObject* hObj)
{
	cPbcastNode *node = (cPbcastNode *)hObj;
	return mEndpoint->Equals(node->mEndpoint);
}

bool cPbcastNode::LessThan(cHashObject* hObj)
{
	return mEndpoint->LessThan( ((cPbcastNode *)hObj)->mEndpoint );
}


/*
 * cPbcastNode::HeartbeatOlder()
 *
 * Purpose:	Returns true if this node has an older heartbeat than the given node.
 * IN:		newNode		-> node to compare heartbeats with.
 * OUT:		-
 * Cond:	-
 * PostCnd:	This nodeinfo is set up
 * Return:	true if this node's heartbeat is older than the given node.
 */
bool cPbcastNode::HeartbeatOlder(cPbcastNode* newNode)
{
	u_short a;
	u_short b;
	a = mHeartbeat;
	b = newNode->mHeartbeat;

	return ( ((a < b) && ((b-a) < MAX_HEARTBEAT/2)) || ((a > b) && ((a-b) > MAX_HEARTBEAT/2)) );
}

/*
 * cPbcastNode::SeqNumOlder()
 *
 * Purpose:	Returns true if seqnum a is older than seqnum b
 * IN:		a, b		-> sequence numbers to compare.
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if a is an older sequence number than b
 */
bool cPbcastNode::SeqNumOlder(u_long a, u_long b)
{
	return ( ((a < b) && ((b-a) < MAX_SEQNUM/2)) || ((a > b) && ((a-b) > MAX_SEQNUM/2)) );
}

/*
 * cPbcastNode::SeqNumDifference()
 *
 * Purpose:	Returns the sequence number difference between older and newer.
 * IN:		older	-> The older sequence number
 *			newer	-> The newer sequence number
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if a is an older sequence number than b
 */
unsigned int cPbcastNode::SeqNumDifference(u_long older, u_long newer)
{
	if(newer > older)
	{
		return newer-older;
	}
	else
	{
		return ((unsigned)(0-older)) + (newer-0);
	}
}